## -*-Tcl-*-
 # ###################################################################
 #  Vince's  Additions - an extension package for Alpha
 # 
 #  FILE: "latexEnvironments.tcl"
 #                                    created: 22/1/98 {9:00:25 am} 
 #                                last update: 11/19/2001 {16:57:49 PM} 
 #  Author: Vince Darley
 #  E-mail: <vince@santafe.edu>
 #    mail: 317 Paseo de Peralta
 #          Santa Fe, NM 87501, USA
 #     www: <http://www.santafe.edu/~vince/>
 #  
 # Some copyright (c) 1998  Vince Darley
 # 
 # See the file "license.terms" for information on usage and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # Many procs written/adapted from Pierre Basso's.  Many thanks for
 # the great improvements.
 # ###################################################################
 ##

#This item is off by default now.  Add it to your TeXPrefs.tcl if desired.
#set TeXbodyOptions(enumerate) "\[a|i\]"
set TeXbodyOptions(list) "\{\}\{\}"
set TeXbodyOptions(figure) "\[tbp\]"
set TeXbodyOptions(floatingfigure) "\{width\}"

#  Embeddable environments  #

proc TeX::Figure {} {
	set fig_types [list "Normal" "Floating" \
		"2 side-by-side" "3 side-by-side" "4 side-by-side" \
		"2, one above the other" \
		"4 in a 2x2 block" \
		"6 with 3 across, 2 down" \
		"6 with 2 across, 3 down" \
		"other" \
		]
	set fig [listpick -p "Pick a figure type to insert:" $fig_types]
	if {$fig == ""} { return "" }

	global TeXbodyOptions
	if {$fig == "Floating"} {
		set t $TeXbodyOptions(floatingfigure)
	} else {
		set t $TeXbodyOptions(figure)
	}
	append t "\n\t"

	switch $fig {
		"Normal" -
		"Floating" {
			append t "\\centerline\{\\includegraphics\[shape,orientation\]"
			append t "\{graphics file\}\}\r\t"
			append t "\\caption\[short title for t.o.f.\]\{caption\}\r\t"
			append t "\\protect[TeX::label fig]"
			if {$fig == "Floating"} {
				text::replace {\begin{figure}} {\begin{floatingfigure}} 0
				text::replace {\end{figure}} {\end{floatingfigure}} 1
				TeX::RequirePackage floatflt
			}
		}
		"2 side-by-side" {
			append t [TeX::_subFigure 2 1]
		}
		"3 side-by-side" {
			append t [TeX::_subFigure 3 1]
		}
		"4 side-by-side" {
			append t [TeX::_subFigure 4 1]
		}
		"2, one above the other" {
			append t [TeX::_subFigure 1 2]
		}
		"4 in a 2x2 block" {
			append t [TeX::_subFigure 2 2]
		}
		"6 with 3 across, 2 down" {
			append t [TeX::_subFigure 3 2]
		}
		"6 with 2 across, 3 down" {
			append t [TeX::_subFigure 2 3]
		}
		"other" {
			set w [prompt "Number of subfigures, horizontally" "2"]
			set h [prompt "Number of subfigures, vertically" "2"]
			append t [TeX::_subFigure $w $h]
		}
		
	}
	return $t	
}


# This procedure is an improved version of tcl programming 
# of tabular environment. 
#
# This TeX::tabular procedure asks for vertical and for horizontal
# lines. But the main Improvements  affect   providing 
# the list of options. When clicking in TeX menu mode 
# the environment "tabular" this procedure asks for a list
# of options given in following way:
#  opt1 opt2 opt3 .....
# opt1, 2, 3, .. are any kind of option allowed by LaTeX
# for environment tabular:
#     c l r c p m b 
# Options >, <, @ and ! are also allowed and you can provide:
#     >c >l >r >p >m >b
#     c< l< r< p< m< b<
#     c@ l@ r@ p@ m@ b@
#     c! l! r! p! m! b!
# Procedure tabular will put left and right braces in case of
# options p, m, b, @, !.
#
# WARNING: space is needed between two options!!!!!!l 
# 
# If the number of options exceed the number of columns
# a message is displayed and the list of options is again asked.
# If the number of options is lesser thanthe number of columns
# the missing options are provided as the last option of the list;
# for example: 5 columns asked and a list of options  "c l r"  with
# vertical line "yes" would create a tabular environment:
# \begin{tabular}{|c|l|r|r|r|}
#
# Pierre BASSO 
# email:basso@lim.univ-mrs.fr
#

proc TeX::BuildTabular {env} {

# ask for number of rows, columns, vertical and horizontal lines and options
	set values [dialog -w 500 -h 400\
	  -t "how many rows? " 10 10 180 30 -e "3" 220 10 240 30 \
	  -t "how many columns? " 10 40 180 60 -e "3" 220 40 240 60 \
	  -t "vertical line (yes/no)? " 10 80 200 100 -e "yes" 220 80 250 100 \
	  -t "horizontal line (yes/no)? " 10 110 200 130 -e "yes" 220 110 250 130 \
	  -t "options:  c l r p m b  >c >l >r >p >m >b  c< l< r< p< m< b<" 10 150 400 170\
	  -t " c@ l@ r@ p@ m@ b@  c! l! r! p! m! b!" 140 170 400 190\
	  -e "c" 220 200 450 230 \
	  -t "position  (b/t or empty)" 10 250 180 270 -e "" 220 250 240 270\
	  -t "width if tabular*" 10 290 180 310 -e "" 220 290 320 310\
	  -b OK 50 350 115 370 -b Cancel 250 350 315 370]
	set cancel [string trim [lindex $values 8]]
	if {$cancel == 1} {beep ; return}

#   search for number of rows, default 3
#   search for number of columns, default 3	
#   
	set numberRows [string trim [lindex $values 0]]
	set numberColumns [string trim [lindex $values 1]]						
	if {![is::PositiveInteger $numberRows] || ![is::PositiveInteger $numberColumns]} {
		beep
		alertnote "invalid input:  unsigned, positive integer required"
		return
	}

# ask for  vertical lines : default yes
	set vline [string trim [lindex $values 2]]
	if {$vline == "yes" } {set vline "|" } else { set vline "" }

# tabular options 
	set options [string trim [lindex $values 4]]
	set numberOpts [llength $options]
	set arg "{$vline"
	for {set j 0} {$j < $numberColumns} {incr j} {
		if {$j < $numberOpts} {
			set optCol [lindex $options $j]
		}
		if {$optCol == "c" || $optCol == "l" || $optCol == "r"} {
		   append arg "$optCol$vline"
	       continue
		}
		if {$optCol == "p" || $optCol == "m" || $optCol == "b"\
		    || $optCol == "@" || $optCol == "!"} {
		    append arg "$optCol\{\}$vline"
		    continue
		}
		if {$optCol == ">c" || $optCol == ">l" || $optCol == ">r"} {
		    append arg  ">\{\}"	
		    set secondopt [string index $optCol 1]
		    append arg "$secondopt$vline"
		    continue
		}
	    if {$optCol == ">p" || $optCol == ">m" || $optCol == ">b"} {
		    append arg  ">\{\}"	
		    set secondopt [string index $optCol 1]
		    append arg "$secondopt\{\}$vline"
		    continue
		}
		if {$optCol == "c<" || $optCol == "l<" || $optCol == "r<"} {
		    set secondopt [string index $optCol 0]
		    append arg "$secondopt<\{\}$vline"
		    continue
		}
		if {$optCol == "p<" || $optCol == "m<" || $optCol == "b<"} {
		    set secondopt [string index $optCol 0]
		    append arg "$secondopt\{\}<\{\}$vline"
		    continue
		}
		if {$optCol == "c@" || $optCol == "l@" || $optCol == "r@"} {
		    set secondopt [string index $optCol 0]
		    append arg "$secondopt@\{\}"
		    continue
		}
		if {$optCol == "p@" || $optCol == "m@" || $optCol == "b@"} {
		    set secondopt [string index $optCol 0]
		    append arg "$secondopt\{\}@\{\}"
		    continue
		}
		if {$optCol == "c!" || $optCol == "l!" || $optCol == "r!"} {
		    set secondopt [string index $optCol 0]
		    append arg "$secondopt!\{\}"
		    continue
		}
		if {$optCol == "p!" || $optCol == "m!" || $optCol == "b!"} {
		    set secondopt [string index $optCol 0]
		    append arg "$secondopt\{\}!\{\}"
		    continue
		}
	}
	append arg "}"
	append t $arg "\r\t"
# set horizontal lines 
	set hline [string trim [lindex $values 3]]
	if {$hline == "yes"} {set hline "\\hline" } else {set hline "" }
	set body "$hline\r\t"
	for {set i 1} {$i <= $numberRows} {incr i} {
		append body "[buildRow $numberColumns]"
		append body "  \\\\\r\t$hline\r\t"
	}
	append t $body " "
# set width if tabular*
	if {$env == "tabular*"} {
		set width [string trim [lindex $values 6]]
		if {$width == ""} {set width ""}
		append t "\{" $width "\}" 
	}	
  	
# set position
	set position [string trim [lindex $values 5]]
	if {$position != ""} {
		append t "\[" $position "\]" 
	}
	return $t
}

#####################  lists  ##########################
 
#
# Build lists. 
# This procedure is called for building environments itemize, enumerate,
# description, list and trivlist.
#
# Pierre BASSO 
# email:basso@lim.univ-mrs.fr

proc TeX::BuildList {env} {
	prompt::simple \
	  [list "how many items?" numberitems 3 N] 
	global TeXbodyOptions
	if [info exists TeXbodyOptions($env)] {
		set t $TeXbodyOptions($env)
	}
	append t "\n\t"
	
	for {set j 1} {$j <= $numberitems} {incr j} {
		if {$env == "description" || $env == "trivlist"} {
			append t "\\item\[\] "
		} else {
			append t "\\item "
		}
		if {$j < $numberitems} {append t "\r\r\t"}
	}
	return $t
}

#  Boxes  #

proc TeX::parbox {} {
	prompt::simple \
	  [list "position (optional)?" position ""] \
	  [list "height (optional)?" height ""] \
	  [list "inner position (optional)?" innerpos ""] \
	  [list "width?" width 3in]

	if {$position != ""} {
		append res "\[$position\]"
	}
	if {$height != ""} {
		append res "\[$height\]"
	}
	if {$innerpos != ""} {
		append res "\[$innerpos\]"
	}
	if {$width == ""} { set width "required width" }
	append res "\{$width\}\{\}"
	return $res	
}  

proc TeX::boxes {} {
	prompt::simple \
	  [list "width (optional)?" width ""] \
	  [list "position (optional)?" position ""]
	
	if {$width != ""} {
		append res "\[$width\]"
	} 
	if {$position != ""} {
		append res "\[$position\]"
	}
	append res "\{\}"
	return $res		
}

proc TeX::raisebox {} {
	prompt::simple \
	  [list "lift?" lift ""] \
	  [list "depth (optional)?" depth ""] \
	  [list "height (optional)?" height ""]
	
	if {$lift == ""} {set lift "required lift"}
	append res "\{$lift\}"
	if {$depth != ""} {
		append res "\[$depth\]"
	}
	if {$height != ""} {
		append res "\[$height\]"
	}
	
	append res "\{\}"
	return $res	
	
}

proc TeX::sbox {} {
	prompt::simple \
	  [list "command?" command ""] 
	if {$command == ""} {set command "required command"}
	append res "\{$command\}\{\}"
	return $res	
}	

proc TeX::savebox {} {
	prompt::simple \
	  [list "command?" command ""] \
	  [list "width (optional)?" width ""] \
	  [list "position (optional)?" position ""] 
	
	if {$command == ""} {set command "required command"}
	append res "\{$command\}"
	
	if {$width != ""} {
		append res "\[$width\]"
	}
	if {$position != ""} {
		append res "\[$position\]"
	}
	
	append res "\{body\}"
	return $res	
	
}

proc TeX::rule {} {
	prompt::simple \
	  [list "lift (optional)?" lift ""] \
	  [list "width?" width ""] \
	  [list "height?" height ""]
	if {$lift != ""} {
		append res "\[$lift\]"
	} 
	
	if {$width == ""} {set width "required width"}
	append res "\{$width\}"
	
	if {$height == ""} { set height "required height" }
	append res "\{$height\}rule body"

	return $res
}

proc TeX::minipage {} {
	prompt::simple \
	  [list "position (optional)?" position ""] \
	  [list "height (optional)?" height ""] \
	  [list "inner position (optional)?" innerpos ""] \
	  [list "width?" width 3in]
	if {$position != ""} {
		append res "\[$position\]"
	}
	if {$height != ""} {
		append res "\[$height\]"
	}
	if {$innerpos != ""} {
		append res "\[$innerpos\]"
	}
	if {$width == ""} { set width "required width" }
	append res "\{$width\}\r\tminipage body"
	return $res
}

#  Maths  #

#	
# Pierre BASSO 
# email:basso@lim.univ-mrs.fr
#

#
# All the types of matrix
# 
#
proc TeX::matrix {} {
	prompt::simple \
	  [list "how many rows?" numberRows 2 N] \
	  [list "how many columns?" numberColumns 2 N]
	set body "\r"
	# build matrix
	for {set i 1} {$i <= $numberRows} {incr i} {
		append body "\t[buildRow $numberColumns]  "
		if {$i != $numberRows} { append body "\\\\\r" }
	}

	return $body
}

#
# Alignment at a single place of mathematical formula
#
proc TeX::align {} {
	prompt::simple \
	  [list "how many equations?" numberRows 2 N] 
	
	#  align is a tabular with two columns
	set numberColumns 2							
	# build alignment
	set res "\r"
	for {set i 1} {$i <= $numberRows} {incr i} {
		append res "\t &  [TeX::label eq]"
		if {$i != $numberRows} { append res "\\\\\r" }
	}
	return $res	
}


#
# Alignment at several places of mathematical formula
#
proc TeX::alignat {} {
	prompt::simple \
	  [list "how many equations?" numberRows 2 N] \
	  [list "how many alignments?" numberColumns 2 N]

	append res "\{$numberColumns\}\r"
	set numberColumns [expr $numberColumns*2 - 1]
	# build alignment
	for {set i 1} {$i <= $numberRows} {incr i} {
		set j 1
		while {$j < $numberColumns} {
			append res "\t & "
			incr j
		}
		append res "  [TeX::label eq]"
		if {$i != $numberRows} { append res "\\\\\r" }
	}
	return $res
}


#  embeddable proc helpers  #

## 
 # -------------------------------------------------------------------------
 # 
 # "TeX::_subFigure" --
 # 
 #  This is a helper, it is only called form the above TeX::Figure proc.
 # -------------------------------------------------------------------------
 ##
proc TeX::_subFigure {w h} {
	TeX::RequirePackage subfigure
	set t "\\centering\r\t"
	set wnum [lindex {x "" two three four five six seven} $w]
	for {set hh $h} {$hh >0} {incr hh -1} {
		for {set ww $w} {$ww >0} {incr ww -1} {
			append t "\\subfigure\[subfig caption\]\{[TeX::label fig]%\r\t"
			append t "\t\\includegraphics\[width=\\figs${wnum}\]"
			append t "\{graphics file\}\}"
			if {$ww != 1} {
				append t "\\goodgap${wnum}\r\t"
			} else {
				if {$hh != 1} {
					append t "\\\\\r\t"
				} else {
					append t "%\r\t"
				}
			}
		}
	}
	append t "\\caption\[short caption for t.o.f.\]\{caption\}\r\t"
	append t "[TeX::label fig]"
}

#--------------------------------------------------------------------------
#  Environments: 
#--------------------------------------------------------------------------

proc enumerate {} {
	set envName "enumerate"
	prompt::var "$envName:  how many items?" numberItems 3 \
	  is::PositiveInteger 1 "invalid input:  unsigned, postive integer required"
	if {$numberItems} {
		set body "\t\\item  type first item"
		for {set i 1} {$i < $numberItems} {incr i} {
			append body "\r\r\t\\item  "
		}
		append body "\r"
	} else {
		set body "\t\r"
	}
	insertEnvironment $envName "" $body
}
proc itemize {} {
	set envName "itemize"
	prompt::var "$envName:  how many items?" numberItems 3 \
	  is::PositiveInteger 1 "invalid input:  unsigned, postive integer required"
	if {$numberItems} {
		set body "\t\\item  type first item"
		for {set i 1} {$i < $numberItems} {incr i} {
			append body "\r\r\t\\item  "
		}
		append body "\r"
	} else {
		set body "\t\r"
	}
	insertEnvironment $envName "" $body
}
proc description {} {
	set envName "description"
	prompt::var "$envName:  how many items?" numberItems 3 \
	  is::PositiveInteger 1 "invalid input:  unsigned, postive integer required"
	if {$numberItems} {
		set body "\t\\item\[label\]  "
		for {set i 1} {$i < $numberItems} {incr i} {
			append body "\r\r\t\\item\[\]  "
		}
		append body "\r"
	} else {
		set body "\t\r"
	}
	insertEnvironment $envName "" $body
}
proc thebibliography {} {
	set envName "thebibliography"
	prompt::var "$envName:  how many items?" numberItems 3 \
	  is::PositiveInteger 1 "invalid input:  unsigned, postive integer required"
	set arg "{9length of the key field}"
	if {$numberItems} {
		if {$numberItems > 9} {set arg "{99length of the key field}"}
		set body "\t\\bibitem{}  "
		for {set i 1} {$i < $numberItems} {incr i} {
			append body "\r\r\t\\bibitem{}  "
		}
		append body "\r"
	} else {
		set body "\t\r"
	}
	insertEnvironment $envName $arg $body
}
proc slide {} { doWrapEnvironment "slide" }
proc overlay {} { doWrapEnvironment "overlay" }
proc note {} { doWrapEnvironment "note" }
proc figure {} {
	global TeXmodeVars
	set envName "figure"
	set envArg "tbp"
	set arg "\[${envArg}Modify this argument?  (t=top; b=bottom; p=page; h=here; !=try harder)\]"
	set theIndentation [text::indentString [getPos]]
	append arg "\r$theIndentation\t\\centering"
	set body ""
	set macro [lindex $TeXmodeVars(boxMacroNames) 0]
	if { $macro != "" } {
		set restOfMacros [lrange $TeXmodeVars(boxMacroNames) 1 end]
		if { ![llength $restOfMacros] } {
			append body "\t\\$macro{}\r"
		} else {
			set cmd [list prompt "Choose a box macro:"]
			lappend cmd $macro "" 
			foreach boxMacroName $TeXmodeVars(boxMacroNames) {
				lappend cmd $boxMacroName
			}
			catch $cmd macro
			if {$macro == "cancel"} {
				message "operation canceled"
				return
			} elseif {$macro == ""} {
				# do nothing
			} else {
				append body "\t\\$macro{}\r"
			}
		}
	}
	append body "\t\\caption{}\r"
	append body "\t[TeX::label fig]\r"
	if { $macro == "" } {
		wrapEnvironment $envName $arg $body
	} else {
		insertEnvironment $envName $arg $body
	}
}
proc table {} {
	set envName "table"
	set envArg "tbp"
	set arg "\[${envArg}Modify this argument?  (t=top; b=bottom; p=page; h=here; !=try harder)\]"
	set theIndentation [text::indentString [getPos]]
	append arg "\r$theIndentation\t\\centering"
	# The following statement puts the caption at the top:
	append arg "\r$theIndentation\t\\caption{}"
	# The following statement puts the caption at the bottom:
# 	set body "\t\\caption{}\r"
	append body "\t[TeX::label tbl]\r"
	wrapEnvironment $envName $arg $body
}
proc buildRow {jmax} {
	set txt ""
	for {set j 1} {$j < $jmax} {incr j} {
		append txt " & "
	}
	return $txt
}
proc tabular {} {
	set envName "tabular"
	prompt::var "$envName:  how many rows?" numberRows 3 \
	  is::PositiveInteger 1 "invalid input:  unsigned, postive integer required"
	prompt::var "$envName:  how many columns?" numberColumns 3 \
	  is::PositiveInteger 1 "invalid input:  unsigned, postive integer required"
	set arg "{|"
	for {set j 1} {$j <= $numberColumns} {incr j} {
		append arg "cmodify this argument?|"
	}
	append arg "}"
	set body "\t\\hline\r"
	for {set i 1} {$i <= $numberRows} {incr i} {
		append body "\t[buildRow $numberColumns]"
		append body "  \\\\\r\t\\hline\r"
	}
	insertEnvironment $envName $arg $body
}
proc verbatim {} { doWrapEnvironment "verbatim" }
proc quote {} { doWrapEnvironment "quote" }
proc quotation {} { doWrapEnvironment "quotation" }
proc verse {} { doWrapEnvironment "verse" }
proc flushleft {} { doWrapEnvironment "flushleft" }
proc center {} { doWrapEnvironment "center" }
proc flushright {} { doWrapEnvironment "flushright" }
proc minipage {} {
    set arg "\[\]{}"
    wrapEnvironment "minipage" $arg ""
    message "enter the position \[b|c|t\] of the minipage, then the width"
}

#--------------------------------------------------------------------------
#  Math Environments: 
#--------------------------------------------------------------------------

proc math {} { checkMathMode "math" 0; doWrapEnvironment "math" }
proc equation* {} { checkMathMode "equation*" 0; doWrapEnvironment "equation*" }
proc subequations {} {
    checkMathMode "subequations" 0;
    wrapEnvironment "subequations" "[TeX::label eq]" ""
}
    
proc displaymath {} {
    checkMathMode "displaymath" 0
    global TeXmodeVars
    if { $TeXmodeVars(useBrackets) } {
	doWrapStructure {\[} {} {\]}
    } else {
	doWrapEnvironment "displaymath"
    }
}

proc mathEnvironment {envName} {
    checkMathMode $envName 0
    set body "\t[TeX::label eq]\r"
    if {[wrapEnvironment $envName "" $body]} {
	set msgText "equation wrapped"
    } else {
	set msgText "enter equation"
    }
    message $msgText
}

proc TeXmathenv {envName} {
    checkMathMode "$envName" 0
    
    prompt::var "$envName:  how many rows?" numberRows 3 \
      is::PositiveInteger 1 "invalid input:  unsigned, positive integer required"
    
    switch $envName {
	"eqnarray*" {
	    set row "\t[buildRow 3]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r"
	}
	"eqnarray" {
	    set row "\t[buildRow 3]\r\t[TeX::label eq]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r"
	}
	"flalign*" - "align*" - "aligned" {
	    prompt::var "$envName:  how many alignments?" numberColumns 1 \
	      is::PositiveInteger 1 "invalid input:  unsigned, positive integer required"
	    set numberColumns [expr $numberColumns*2 ]
	    set row "\t[buildRow $numberColumns]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r"
	}
	"flalign" - "align" {
	    prompt::var "$envName:  how many alignments?" numberColumns 1 \
	      is::PositiveInteger 1 "invalid input:  unsigned, positive integer required"
	    set numberColumns [expr $numberColumns*2]
	    set row "\t[buildRow $numberColumns]\r\t[TeX::label eq]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r"
	}
	"gather*" - "multline*" - "gathered" - "split"  {
	    set row "\t[buildRow 1]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r"
	}
	"multline" {
	    set row "\t[buildRow 1]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r\t[TeX::label eq]\r"
	}
	"gather" {
	    set row "\t[buildRow 1]\r\t[TeX::label eq]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r"
	}
	"cases" {
	    set row "\t[buildRow 2]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r\t[TeX::label eq]\r"
	}
	"array" {
	    prompt::var "$envName:  how many columns?" numberColumns 3 \
	      is::PositiveInteger 1 "invalid input:  unsigned, positive integer required"
	    set arg "{"
	    for {set j 1} {$j <= $numberColumns} {incr j} {
		append arg "c"
	    }
	    append arg "Modify this argument?  (c=center; l=left; r=right; p{width}; a{text})}"
	    set row "\t[buildRow $numberColumns]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r"
	    insertEnvironment $envName $arg $body
	    return
	}
	"alignat" {
	    prompt::var "$envName:  how many columns?" numberColumns 3 \
	      is::PositiveInteger 1 "invalid input:  unsigned, positive integer required"
	    set arg "{$numberColumns}"
	    set numberColumns [expr $numberColumns*2]
	    set row "\t[buildRow $numberColumns]\r\t[TeX::label eq]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r"
	    insertEnvironment $envName $arg $body
	    return
	}
	"alignat*" - "alignedat" {
	    prompt::var "$envName:  how many columns?" numberColumns 3 \
	      is::PositiveInteger 1 "invalid input:  unsigned, positive integer required"
	    set arg "{$numberColumns}"
	    set numberColumns [expr $numberColumns*2]
	    set row "\t[buildRow $numberColumns]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r"
	    insertEnvironment $envName $arg $body
	    return
	}
	"subarray" {
	    prompt::var "$envName:  how many columns?" numberColumns 1 \
	      is::PositiveInteger 1 "invalid input:  unsigned, positive integer required"
	    set arg "{"
	    for {set j 1} {$j <= $numberColumns} {incr j} {
		append arg "c"
	    }
	    append arg "Modify this argument?  (c=center; l=left; r=right; p{width}; a{text})}"
	    set row "\t[buildRow $numberColumns]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r"
	    insertEnvironment $envName $arg $body
	    return
	}
	"matrix" - "pmatrix" - "bmatrix" - "Bmatrix" - "vmatrix" - "Vmatrix" - \
	  "smallmatrix" {
	    prompt::var "$envName:  how many columns?" numberColumns 3 \
	      is::PositiveInteger 1 "invalid input:  unsigned, positive integer required"
	    set row "\t[buildRow $numberColumns]"
	    for {set i 1} {$i < $numberRows} {incr i} {
		append body $row
		append body "  \\\\\r"
	    }
	    append body $row
	    append body "\r"
	}
    }
    insertEnvironment $envName "" $body
}

